#!/bin/bash
#
# Copyright (c) 2015 by Cisco Systems, Inc.
#
# Name        : pre_upgrade.sh
# BUILD_TIME  : VSMS_release
# Description : Script to run before upgrade of VSM rpms.
# Input       : none
# Action      : restores system activities after upgrade
# Output      : /tmp/restart_after_upgrade script which is consumed by post_upgrade script


RESTART_AFTER_UPGRADE=/tmp/restart_after_upgrade 
LOG_PRE_UPG_KERNEL=/usr/BWhttpd/upgrade/server/pre-upgrade-kernel-ver

function LOG {
    #local dest_log=$1
    local msg=$1

    local log_prefix
    local log_date
    local output

    log_date=$(date +"%F %T")
    log_prefix="$log_date.??? [ $(basename $0)($$) ] "

    output=$(echo "$msg" | sed -e "s/^/$log_prefix/g" \
                               -e "s/\\\n/\\\n$log_prefix/g")

    printf "$output\n" #>> $dest_log 
}


function stopPacemaker {

    if `service pacemaker status > /dev/null 2>&1` ;
    then
        echo pacemaker : was running, stopping for upgrade 
        echo "service pacemaker start" >> $RESTART_AFTER_UPGRADE        
    else 
        return
    fi

    # stop pacemaker incase it is running
    # check pacemaker resource status

    if [[ ! `which  crm_resource` ]] ;
    then 
        LOG "crm_resource not found. returning"
        return
    fi

    pacemaker_status=`crm_resource -List | grep -i 'fail'`
    running=`/sbin/service pacemaker status | grep 'running'`
    if [ "x$pacemaker_status" = "x" -a "x$running" != "x" ]
    then
        gracefulStopPacemaker
    else
        LOG "resetting the fail count and cleaning up failed actions prior to gracefully stopping pacemaker" 
        #reset the failcount 
        crm_failcount -v 0 -r vsom
        #cleanup failed actions
        crm_resource --cleanup --resource vsom

        gracefulStopPacemaker
    fi

}

function forceStopPacemaker {
    LOG "\nForce shutting down pacemaker. Noticed some pacemaker resources were down, $pacemaker_status" 
    /usr/bin/killall -9 corosync \
                        fenced \
                        dlm_controld \
                        gfs_controld \
                        pacemakerd \
                        cib \
                        stonithd \
                        lrmd \
                        attrd pengine crmd 
    /bin/rm -rf         /var/run/corosync.pid 
                        /var/run/fenced.pid \
                        /var/run/dlm_controld.pid \
                        /var/run/cman.pid \
                        /var/run/gfs_controld.pid \
                        /var/run/pacemakerd.pid \
                        /var/run/cman*
}

function forceKillStalePacemakerProcesses {
    for pid in $(/usr/bin/pgrep -f 'corosync|fenced|dlm_controld|gfs_controld|pacemakerd|cib|stonithd|lrmd|attrd|pengine|crmd')
    do
        pname=`/bin/ps -p $pid | /bin/awk '{ print $4 }' | tail -n +2`
        LOG "\nKilling Process $pname with pid $pid"
        `/bin/kill -9 $pid`
    done
    /bin/rm -rf         /var/run/corosync.pid 
                        /var/run/fenced.pid \
                        /var/run/dlm_controld.pid \
                        /var/run/cman.pid \
                        /var/run/gfs_controld.pid \
                        /var/run/pacemakerd.pid \
                        /var/run/cman*
}

function gracefulStopPacemaker {
    LOG "stopping pacemaker service gracefully"
    /sbin/service pacemaker stop &
    stop_pid="$!"
    #wait upto 60secs to gracefully stop the service before force killing the process
    while :; do
        pname=`/bin/ps -p $stop_pid | /bin/awk '{ print $4 }' | tail -n +2`
        if  [ -z $pname ] ; then
            LOG "thru with stopping the process $stop_pid after $CNT secs"
            #incase any pacemaker processes are still seen lingering around, force kill those
            forceKillStalePacemakerProcesses  
            break;
        fi
        #If it took more than a minute, force kill the process
        if [[ "$CNT" -ge "60" ]]; then
            LOG "waited upto $CNT secs and the pacemaker service is not yet stopped"
            forceStopPacemaker
            break;
        fi
        ((CNT++))
        sleep 1
    done
}


function disable_service {
    SERVICE=$1
    [[ -z SERVICE ]] && return
    if `chkconfig --list | egrep "^$SERVICE " > /dev/null 2>&1` ;
    then
        LOG "$SERVICE : found.  disabling"
        service $SERVICE stop
        chkconfig --del $SERVICE
    else
        LOG "$SERVICE : already disabled"
    fi
}

function stop_service_for_upgrade {
    SERVICE=$1
    [[ -z SERVICE ]] && return
    if `service $SERVICE status > /dev/null 2>&1 ` ;
    then
        LOG "$SERVICE : was running, stopping for upgrade"
        service $SERVICE stop
        echo "service $SERVICE start" >> $RESTART_AFTER_UPGRADE
    else
        LOG "$SERVICE : was not running"
    fi
}

function stop_gpac_process {
    RPM_NAME=$1
    if `rpm -qa | grep $RPM_NAME > /dev/null 2>&1` ;
    then
        LOG "Uninstalling $RPM_NAME"
        rpm -e --nodeps $RPM_NAME
        rm -rf /usr/BWhttpd/logs/gpac.log
	if [ -f /usr/BWhttpd/bin/gpac_script/Kill_all_fk_cisco.sh ]
	then
		/usr/BWhttpd/bin/gpac_script/Kill_all_fk_cisco.sh
	fi
        rm -rf /usr/BWhttpd/bin/gpac_script
        rm -rf /var/gpac
        rm -rf /usr/BWhttpd/bin/fk-cisco.exe
    else
        LOG "$RPM_NAME is not installed"
    fi
	
    rm -rf /usr/BWhttpd/logs/CiscoCast.log
    if [ -f /usr/BWhttpd/bin/CiscoCast_script/Kill_all_fk_cisco.sh ]
    then
	/usr/BWhttpd/bin/CiscoCast_script/Kill_all_fk_cisco.sh
    fi

    rm -rf /usr/BWhttpd/bin/CiscoCast_script
    rm -rf /var/CiscoCast/
	    
    rm -rf /usr/BWhttpd/logs/DashCast.log
    if [ -f /usr/BWhttpd/bin/DashCast_script/Kill_all_fk_cisco.sh ]
    then
	/usr/BWhttpd/bin/DashCast_script/Kill_all_fk_cisco.sh
    fi
    rm -rf /usr/BWhttpd/bin/DashCast_script/
    rm -rf /var/DashCast/
}

function uninstall_rpm {
    RPM_NAME=$1
    if `rpm -qa | grep $RPM_NAME > /dev/null 2>&1` ;
    then
        LOG "Uninstalling $RPM_NAME"
        rpm -e --nodeps $RPM_NAME
    else
        LOG "$RPM_NAME is not installed"
    fi
}

function make_or_convert_conf_links {
    USR_CONF=/usr/BWhttpd/conf
    VAR_CONF=/var/BWhttpd/conf
    # If there is a regular file by the name of conf, give an error
    # Rename the file to conf.bad
    if [ -e $VAR_CONF ] && [ -f $VAR_CONF ]
    then
        LOG "Error: $VAR_CONF is not a directory"
        LOG "Renaming /var/BWhttpd/conf to /var/BWhttpd/conf.bad"
        mv $VAR_CONF /var/BWhttpd/conf.bad
    fi

    # Unconditionally create a directory in /var/BWhttpd, it is ok even if
    # exists
    mkdir -p $VAR_CONF

    # If /usr/BWhttpd/conf exists, copy the files to /var/BWhttpd/conf and
    # then remove the usr's conf directory
    if [ -e $USR_CONF ]
    then
        if [ -d $USR_CONF ]
        then
	    # Take a count of how many files are there in the usr's conf
            pre_count=$(ls -l $USR_CONF/ | wc -l)
            cp -r $USR_CONF/* $VAR_CONF
            post_count=$(ls -l $VAR_CONF | wc -l)
            # Compare to make sure we have the same number of files
            if [ "$pre_count" -eq "$post_count" ]
            then
	 	LOG "Removing $USR_CONF"
            	rm -rf $USR_CONF
            else
                LOG "Number of files did not match in /usr/BWhttpd/conf "
                LOG "and /var/BWhttpd/conf"
                return 1
            fi
        elif [ -f $USR_CONF ]
        then
            LOG "Error: /usr/BWhttpd/conf is a regular file"
            LOG "Copying /usr/BWhttpd/conf to a /var/BWhttpd/conf/conf.bad"
            mv $USR_CONF $VAR_CONF/conf.bad
        fi
    fi

    # Finally, create a soft link in usr to /var/BWhttpd/conf
    LOG "Making $USR_CONF a link to $VAR_CONF"
    ln -sf $VAR_CONF $USR_CONF
    if [ -L $USR_CONF ]
    then
       LOG "Successfully created a link to /var/BWhttpd/conf"
       return 0
    else
       LOG "Link was not created to /var/BWhttpd/conf"
       return 1
    fi
}
function log_current_kernel_ver {
    rpm -qa | grep kernel > $LOG_PRE_UPG_KERNEL
    LOG "Preupgrade kernel is $(< $LOG_PRE_UPG_KERNEL)"
}

function copy_onvif_dev_xml {
    # Unconditionally create a directory in /var/BWhttpd, it is ok even if
   # exists
    VAR_ONVIF=/var/onvif
    mkdir -p $VAR_ONVIF
    rm -rf $VAR_ONVIF/*
    LOG "Copying onvif dev cap xml"
    grep -rl "Onvif Devices" /usr/BWhttpd/conf/devices/ --include=*.xml*  | xargs --no-run-if-empty -d "\n"  cp -p -t /var/onvif/
    LOG "Done."
}

function backup_openssl_cnf {

   echo "Pre upgrade : Backup old opensslAllowedCipher.cnf"
   CONFDIR=/usr/BWhttpd/conf
       OPENSSL_CONF=$CONFDIR/opensslAllowedCipher.cnf
   
   if [  -e $OPENSSL_CONF ]
       then
       cp -p -rf $OPENSSL_CONF /tmp/
       echo "Pre upgrade : Backup old opensslAllowedCipher.cnf completed"
   else
       echo "Pre upgrade : Old opensslAllowedCipher.cnf not present in $CONFDIR"
   fi  
}

function backup_failover_cnf {

   echo "Pre upgrade : Backup old failoverconf.txt"
   CONFDIR=/usr/BWhttpd/conf
   FAILOVER_CONF=$CONFDIR/failoverconf.txt
   
   if [  -e $FAILOVER_CONF ]
       then
       cp -p -rf $FAILOVER_CONF /tmp/
       echo "Pre upgrade : Backup old failoverconf.txt completed"
   else
       echo "Pre upgrade : Old failoverconf.txt not present in $CONFDIR"
   fi  
}

function update_etc_issue_file()
{
	relStr=`head -n 1 /etc/issue`
	if [[ $relStr == *"Red Hat Enterprise Linux Server release"* ]]; then
	    echo "/etc/issue is Okay"
	else
	    echo "correcting /etc/issue"
	    release_name=`head -n 1 /etc/redhat-release`
	    echo "release_name : $release_name"
	    sed -i "1s/^/${release_name}\n/" /etc/issue
        fi
}


# main

rm -f $RESTART_AFTER_UPGRADE

update_etc_issue_file

log_current_kernel_ver
stopPacemaker
disable_service           cups
stop_service_for_upgrade  ntpd
uninstall_rpm             Cisco_VSM_Hotfix-7.6.0-001
stop_gpac_process         Cisco_VSMGpac-7.9.0
copy_onvif_dev_xml
backup_openssl_cnf
backup_failover_cnf
RC=0
if [ -e /etc/redhat-release ]
then
    make_or_convert_conf_links
    RC=$?
fi


[[ -e $RESTART_AFTER_UPGRADE ]] && chmod u+x $RESTART_AFTER_UPGRADE && RC=$?

exit $RC




